#!/bin/bash
###############################################################################
# Author        : onlyshenmin
# Created Time  : 2019-05-29 15:10:38
# Last Modified : 2019-05-30 11:25:04
# File Name     : sysinit.sh
# Description   : system initialization
###############################################################################
# color/title setting
set +e
set -o noglob
bold=$(tput bold)
underline=$(tput sgr 0 1)
reset=$(tput sgr0)
red=$(tput setaf 1)
green=$(tput setaf 2)
yellow=$(tput setaf 3)
blue=$(tput setaf 4)
magenta=$(tput setaf 5)
cyan=$(tput setaf 6)
white=$(tput setaf 7)
underline() {
    printf "${underline}${bold}%s${reset}\n" "$@"
}
h1() {
    printf "\n${underline}${bold}${cyan}%s${reset}\n" "$@"
}
h2() {
    printf "\n${underline}${bold}${white}%s${reset}\n" "$@"
}
debug() {
    printf "${white}%s${reset}\n" "$@"
}
info() {
    printf "${white}-> %s${reset}\n" "$@"
}
success() {
    printf "${bold}${green}✔ %s${reset}\n" "$@"
}
error() {
    printf "${red}✖ %s${reset}\n" "$@"
}
warn() {
    printf "${yellow}%s${reset}\n" "$@"
}
bold() {
    printf "${bold}%s${reset}\n" "$@"
}
note() {
    printf "\n${underline}${bold}${blue}Note:${reset} ${blue}%s${reset}\n" "$@"
}
set -e
set +o noglob
item=1
###############################################################################
# 获取当前目录
work_dir="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
packages_dir="${work_dir}/packages"

# 定义通用软件列表
software_list=(
    bash-completion
    bind-utils
    chrony
    cifs-utils
    lrzsz
    net-tools
    nfs-utils
    python2-pip
    rsync
    vim
    tree
    telnet
    unzip
    wget
    iotop
    yum-utils
    gcc 
    gcc-c++ 
    glibc
    make
    autoconf
    openssl
    openssl-devel
    pcre-devel
    pam-devel
    docker-ce
    docker-ce-cli
    containerd.io
    haveged
)

# 检查输入是否为 yes 或 no
check_input_yes_no() {
    read choice
    if [[ $choice != "yes" && $choice != "no" ]]; then
        error "必须输入 yes 或者 no, 请重新输入："
        check_input_yes_no
    fi
}

# 禁用 firewalld
disable_firewalld() {
    h1 "[Step ${item}]: 禁用防火墙 ..."
    let item+=1
    systemctl stop firewalld
    systemctl disable firewalld
}

# 禁用 selinux
disable_selinux() {
    h1 "[Step ${item}]: 禁用selinux ..."
    let item+=1
    if [ "$(getenforce)" != "Disabled" ]; then
        selinux_flag="true"
        setenforce 0
        sed -i "s/^SELINUX=.*$/SELINUX=disabled/g" /etc/selinux/config
    fi

}

# 通过 yum 安装 rpm 格式的软件包
install_software_rpm() {
    cd ${work_dir}
    h1 "[Step ${item}]: 创建通用软件的安装源 ..."
    let item+=1
    if [ $(ls /etc/yum.repos.d/*.repo 2>/dev/null | wc -l) != 0 ]; then
        mkdir /etc/yum.repos.d/tmp -p
        mv /etc/yum.repos.d/*.repo /etc/yum.repos.d/tmp/ -f
    fi
    cat >/etc/yum.repos.d/software.repo <<EOF
[software]
name=software
baseurl=file://${packages_dir}
enabled=1
gpgcheck=0
EOF
    yum clean all
    rm -rf /var/cache/yum
    yum makecache

    h1 "[Step ${item}]: 安装通用软件  ..."
    let item+=1
    yum install "${software_list[@]}" -y

    h1 "[Step ${item}]: 清理通用软件的安装源 ..."
    let item+=1
    yum clean all
    rm -rf /var/cache/yum
    rm /etc/yum.repos.d/software.repo -rf
    if [ -d /etc/yum.repos.d/tmp ]; then
        mv /etc/yum.repos.d/tmp/*.repo /etc/yum.repos.d/ -f
        rm /etc/yum.repos.d/tmp/ -rf
    fi
}

# 直接安装二进制软件
install_software_binary() {
    cd ${work_dir}
    h1 "[Step ${item}]: 安装docker-compose  ..."
    let item+=1
    cp -a ./binary/docker-compose/docker-compose-1.25.4 /usr/local/bin/docker-compose
    chmod +x /usr/local/bin/docker-compose
    cp -a ./binary/docker-compose/docker-compose-completion-1.25.4 /etc/bash_completion.d/docker-compose
    info "安装完成"

    h1 "[Step ${item}]: 验证docker-compose  ..."
    let item+=1
    /usr/local/bin/docker-compose version;
    if [ $? == 0 ];then
        success "docker-compose 验证成功 ！"
    else
        error "docker-compose 验证失败 !!!"
        exit 1
    fi
}

update_docker_conf(){
    h1 "[Step ${item}]: 更新docker配置  ..."
    mkdir /etc/docker  -p
    cat > /etc/docker/daemon.json<<EOF
{
  "data-root": "/opt/docker",
  "log-driver":"json-file",
  "log-opts": {"max-size":"500m", "max-file":"3"},
  "registry-mirrors": ["https://registry.docker-cn.com"],
  "insecure-registries":["https://reg.citms.cn:7443"]
}
EOF
    # systemctl   restart  docker
    echo -e '\n'
    cat /etc/docker/daemon.json
    echo -e '\n'
    sleep 2

}
# 验证软件安装是否成功
verify_software_installed() {
    h1 "[Step ${item}]: 启动haveged;"
    let item+=1
    systemctl enable haveged
    systemctl start  haveged
    info "版本信息: "`haveged --version  |head -1`

    h1 "[Step ${item}]: 启动docker;"
    let item+=1
    systemctl enable docker
    systemctl start docker
    if docker info| head -16 ; then
        success "docker 启动成功 ！"
    else
        error "docker 启动失败 !!!"
        exit 1
    fi
}

# 安装python的docker模块
install_docker_module(){
    h1 "[Step ${item}]: 安装python docker module  ..."
    cd ${work_dir}/docker_module \
    && pip install *.whl
    if pip list | grep -w "docker (4.4.0)" >/dev/null 2>1&  then
        success "python docke module install success  ！"
    else
        error "python docke module error!!!"
        exit 1
    fi  
}

function ulimits(){
h1 "[Step ${item}]: 设置系统limits  ..."
cat > /etc/security/limits.conf <<EOF
* soft noproc 65536
* hard noproc 65536
* soft nofile 65536
* hard nofile 65536
EOF
# centos 7.3 还是 7.4开始， 这个文件有一部分soft 和 nproc 内容，登陆后会被覆盖，/etc/security/limits.conf 不会生效
echo > /etc/security/limits.d/20-nproc.conf 

ulimit -n 65536
ulimit -u 65536

echo -e '\n'
cat /etc/security/limits.conf
echo -e '\n'
sleep 2

#echo "[ulimits 配置] ==> OK"


}


# 修改内核参数，增加缓存区，减少等待时间
# 可以接收更大的包，增加对轻量ddos抗性
function kernel(){
h1 "[Step ${item}]: 设置内核参数  ..."
cat > /etc/sysctl.conf <<EOF
fs.file-max = 65536
net.core.netdev_max_backlog = 32768
net.core.rmem_default = 8388608
net.core.rmem_max = 16777216
net.core.somaxconn = 32768
net.core.wmem_default = 8388608
net.core.wmem_max = 16777216
net.ipv4.conf.all.arp_ignore = 0
net.ipv4.conf.lo.arp_announce = 0
net.ipv4.conf.lo.arp_ignore = 0
net.ipv4.ip_local_port_range = 5000 65000
net.ipv4.tcp_fin_timeout = 30
net.ipv4.tcp_keepalive_intvl = 30
net.ipv4.tcp_keepalive_probes = 3
net.ipv4.tcp_keepalive_time = 300
net.ipv4.tcp_max_orphans = 3276800
net.ipv4.tcp_max_syn_backlog = 65536
net.ipv4.tcp_max_tw_buckets = 5000
net.ipv4.tcp_mem = 94500000 915000000 927000000
net.ipv4.tcp_syn_retries = 2
net.ipv4.tcp_synack_retries = 2
net.ipv4.tcp_syncookies = 1
net.ipv4.tcp_timestamps = 0
net.ipv4.tcp_tw_recycle = 1
net.ipv4.tcp_tw_reuse = 1
EOF
echo -e '\n'
sysctl -p
echo -e '\n'
sleep 2
#echo "[内核 优化] ==> OK"
}

# 确定是否需要重启服务器，彻底禁用selinux需要重启服务器
reboot_whether() {
    if [ "${selinux_flag}" = "true" ]; then
        h1 "[Step ${item}]: 要彻底禁用selinux需要重启服务器，是否重启？ (yes/no)"
        check_input_yes_no
        let item+=1
        if [ "$choice" == "yes" ]; then
            warn "开始重启服务器 ..."
            reboot
        fi
    fi
}

print_info(){
    echo -e '\n'
    success $"    1.安装常用软件[vim telnet net-tools GCC haveged ....];关闭selinux firewalld"
    success $"    2.安装docker1.19.03; docker-compose-1.25"
    success $"    3.更新内核参数优化"
    success $"    4.更新文件句柄限制"
    success $"    5.关闭ssh DNS 反向解析"
    echo -e '\n'
    sleep 2

}
main() {
    disable_firewalld
    disable_selinux
    install_software_rpm
    install_software_binary
    update_docker_conf
    verify_software_installed
    install_docker_module
    ulimits
    kernel
    print_info 
    reboot_whether
}

main
